unit AriUnit;
{ Arithmetic Encoder/Decoder unit.

  reSource (C) 1997 Cascada Design
}


interface
uses
  SysUtils, Classes,
  bit_file_unit;


var
  totals : array[0..258] of integer;

const
  END_OF_STREAM = 256;


type
  TSymbol = record
    low_count: integer;
    high_count: integer;
    scale: integer;
  end;


  TAriEncoder = class
  private
    low, high, underflow_bits: longint;
    outfile: TBitFile;

  public
    constructor Create;
    destructor Destroy; override;

    procedure Flush;
    procedure ConvertIntToSymbol(c: longint; var s: TSymbol);
    procedure GetSymbolScale(var s: TSymbol);
    function ConvertSymbolToInt(count: longint; var s: TSymbol): longint;
    procedure EncodeSymbol(const s: TSymbol);  
  end;


implementation

constructor TAriEncoder.Create;
begin
  inherited Create;

  low := 0;
  high := $FFFF;
  underflow_bits := 0;

  {Debug. The Outfile is created in this class}
  outfile.Create('c:\ctest\ari_out.txt', fmCreate or fmShareExclusive);
end;

destructor TAriEncoder.Destroy;
begin
  {Debug}
  outfile.free;

  inherited Destroy;
end;

{ At the end of the encoding process, there are still significant bits left
in the high and low registers. We output two bits, plus as many underflow
bits as are necessary }
procedure TAriEncoder.Flush;
begin
  outfile.OutputBit(low and $4000);
  while (underflow_bits > 0) do
  begin
    outfile.OutputBit((not low) and $4000);
    dec(underflow_bits);
  end;
end;

procedure TAriEncoder.ConvertIntToSymbol(c: longint; var s: TSymbol);
begin
  s.scale := totals[END_OF_STREAM + 1];
  s.low_count := totals[c];
  s.high_count := totals[c + 1];
end;

procedure TAriEncoder.GetSymbolScale(var s: TSymbol);
begin
  s.scale := totals[END_OF_STREAM + 1];
end;

function TAriEncoder.ConvertSymbolToInt(count: longint; var s: TSymbol): longint;
var
  c: longint;
begin
  c := END_OF_STREAM;
  while (count < totals[c]) do dec(c);;

  s.high_count := totals[c+1];
  s.low_count := totals[c];
  result := c;
end;

procedure TAriEncoder.EncodeSymbol(const s: TSymbol);
var
  range: longint;
begin
  {Rescale high and low for the new symbol}
  range := high-low + 1;
  high := low + ((range * s.high_count) div s.scale -1);
  low := low + ((range * s.low_count) div s.scale);

  {This loop turns out new bits until high and low are far enough apart to
  have stabilized}
  repeat
    { If this test passes, it means that the MSDigits match and can be sent
    to the output stream. }
    if ((high and $8000) = (low and $8000)) then
    begin
      outfile.OutputBit(high and $8000);
      while (underflow_bits > 0) do
      begin
        outfile.OutputBit((not high) and $8000);
        dec(underflow_bits);
      end;
    end

    {If this test passes, the numbers are in danger of underflow because the
    MSDigits don't match, and the 2nd digits are just one apart}

    else if ((low and $4000 <> 0) and (not(high and $4000 <> 0)) ) then
    begin
      inc(underflow_bits);
      low := low and $3FFF;
      high := high or $4000;
    end
    else exit;

    low := low shl 1;
    high := high shl 1;
    high := high or 1;
  until false;
end;


end.
